unit Minesweeper;

interface

	uses
		Dialogs, QDOffscreen;

	function NewBitMap (var theBitMap: BitMap; theRect: Rect): Ptr;
	function Main (d: DCtlPtr; p: ParmBlkPtr; n: integer): integer;

	procedure HandleUpdate (event: EventRecord);
	procedure HandleActivate (event: EventRecord);

implementation

	const
		dNeedTime = $2000;

		kMenuID = 0;

		kKindBeginner = 0;
		kKindIntermediate = 1;
		kKindExpert = 2;
		kKindCustom = 3;

		kTileRevealed = 1;
		kTileNormal = 2;
		kTileMineTriggered = 3;
		kTileMine = 4;
		kTileFlagged = 5;
		kTileNoMine = 6;
		kTilePossibly = 7;

	type
		Score = packed record
				name: Str255;
				time: LongInt;
			end;
		Scores = array[0..2] of Score;

		Tile = packed record
				isMine: Boolean;
				isRevealed: Boolean;
				isTriggered: Boolean;
				flagType: Integer;
				neighbors: Integer;
				floodfill: Boolean;
			end;
		TilePtr = ^Tile;

	type
		Board = packed record
				kind: Integer;
				width: Integer;
				height: Integer;
				highlight: Point;
				tiles: array[0..0] of Tile;
			end;
		BoardPtr = ^Board;

	var
		gHighScores: Scores;
		gModeNames: array[0..2] of Str63;
		gDone: Boolean;
		gGameOver: Boolean;
		gWin: Boolean;
		gGameStarted: Boolean;
		gAlreadyOpen: Boolean;
		gTrackingTile: Boolean;
		gDCE: DCtlPtr;
		gMenu: MenuHandle;
		gMenuID: integer;
		gBoardType: Integer;
		gWait: Cursor;
		gErr: OSErr;
		gWindow: WindowPtr;
		gWindowSize: Point;
		gTiles: Handle;
		gScreenBounds: Rect;
		gTileBitmaps: array[1..7] of BitMap;
		gButtonBitmaps: array[1..8] of BitMap;
		gTileIcons: array[1..7] of CIconHandle;
		gButtonIcons: array[1..8] of CIconHandle;
		gNeighborColors: array[1..8] of RGBColor;
		gButtonMask: BitMap;
		gBackPat, gWhitePat: Pattern;
		gBoard: BoardPtr;
		gCustomWidth: Integer;
		gCustomHeight: Integer;
		gCustomMines: Integer;
		gTimerTicks: Integer;
		gFlagCounter: Integer;
		gBackgroundRgn: RgnHandle;
		gTmpRgn: RgnHandle;
		gColorQD: Boolean;
		gScoresName: Str255;
		gDefaultName: Str255;
		gScoresVRefNum: Integer;

	function NumToolboxTraps: Integer;
	begin
		if NGetTrapAddress($A86E, ToolTrap) = NGetTrapAddress($AA6E, ToolTrap) then
			NumToolboxTraps := $200
		else
			NumToolboxTraps := $400;
	end;

	function GetTrapType (theTrap: Integer): TrapType;
		const
			TrapMask = $800;
	begin
		if BAND(theTrap, TrapMask) > 0 then
			GetTrapType := ToolTrap
		else
			GetTrapType := OSTrap;
	end;

	function TrapAvailable (theTrap: Integer): Boolean;
		var
			tType: TrapType;
	begin
		tType := GetTrapType(theTrap);
		if tType = ToolTrap then
			begin
				theTrap := BAND(theTrap, $7FF);
				if theTrap >= NumToolboxTraps then
					theTrap := $A89F;
			end;
		TrapAvailable := NGetTrapAddress(theTrap, tType) <> NGetTrapAddress($A89F, ToolTrap);
	end;

	procedure InitColor;
		type
			IntPtr = ^Integer;
		const
			_GetCIcon = $AA1E;
			ROM85 = $28E;
	begin
		gColorQD := TrapAvailable(_GetCIcon) and (BAND(IntPtr(ROM85)^, BNOT($3FFF)) = 0);
		gTmpRgn := NewRgn;
	end;

	function GetTile (x, y: Integer): TilePtr;
	begin
		GetTile := @gBoard^.tiles[y * gBoard^.width + x];
	end;

	procedure GetWaitCursor;
		var
			hptr, wptr: Ptr;
			h: CursHandle;
			dispose: Boolean;
	begin
		SetResLoad(false);
		h := GetCursor(watchCursor);
		dispose := (GetHandleSize(Handle(h)) = 0);
		SetResLoad(true);
		h := GetCursor(watchCursor);
		hptr := Ptr(h^);
		wptr := Ptr(@gWait);
		BlockMove(hptr, wptr, sizeof(Cursor));
		if dispose then
			ReleaseResource(Handle(h));
	end;

	function OwnedResourceID (n: integer): integer;
		var
			drvrID: Integer;
			i: integer;
	begin
{$PUSH}
{$V-}
{$R-}
		drvrID := abs(gDCE^.dCtlRefNum) - 1;
		i := (BOR($C000, n + (BSL(drvrID, 5))));
		OwnedResourceID := i;
{$POP}
	end;

	function NewBitMap;
	begin
		with theBitMap, theRect do
			begin
				rowBytes := ((right - left + 15) div 16) * 2;
				baseAddr := NewPtr(rowBytes * (bottom - top));
				bounds := theRect;
				if (MemError <> noErr) then
					NewBitMap := nil
				else
					NewBitMap := baseAddr;
			end;
	end;

	function InRange (x, y: Integer): Boolean;
	begin
		InRange := (x >= 0) and (x < gBoard^.width) and (y >= 0) and (y < gBoard^.height);
	end;

	procedure NewBoard (forceRedraw: Boolean);
		var
			x, y, xd, yd, n: Integer;
			width: Integer;
			height: Integer;
			mines: Integer;
			newWindowSize: Point;
			r: Rect;
			savePort: GrafPtr;
			invalidate: Boolean;
	begin
		if (gBoard <> nil) then
			DisposePtr(Ptr(gBoard));

		invalidate := forceRedraw;
		gGameOver := false;
		gWin := false;
		gTimerTicks := 0;
		gGameStarted := false;

		case gBoardType of
			kKindBeginner: 
				begin
					width := 9;
					height := 9;
					mines := 10;
				end;
			kKindIntermediate: 
				begin
					width := 16;
					height := 16;
					mines := 40;
				end;
			kKindExpert: 
				begin
					width := 30;
					height := 16;
					mines := 99;
				end;
			otherwise
				begin
					if gCustomMines > 3 * gCustomWidth * gCustomHeight div 4 then
						gCustomMines := 3 * gCustomWidth * gCustomHeight div 4;
					width := gCustomWidth;
					height := gCustomHeight;
					mines := gCustomMines;
				end;
		end;

		gCustomWidth := width;
		gCustomHeight := height;
		gCustomMines := mines;

		gFlagCounter := mines;

		gBoard := BoardPtr(NewPtrClear(SizeOf(Board) + width * height * SizeOf(Tile)));
		gBoard^.width := width;
		gBoard^.height := height;
		gBoard^.kind := gBoardType;

		while (mines > 0) do
			begin
				x := Abs(Random) mod width;
				y := Abs(Random) mod height;
				if (gBoard^.tiles[y * width + x].isMine = false) then
					begin
						gBoard^.tiles[y * width + x].isMine := true;
						mines := mines - 1;
					end;
			end;

		for y := 0 to height - 1 do
			for x := 0 to width - 1 do
				begin
					if (gBoard^.tiles[y * width + x].isMine = false) then
						begin
							n := 0;
							for yd := -1 to 1 do
								for xd := -1 to 1 do
									begin
										if (InRange(x + xd, y + yd)) then
											if (gBoard^.tiles[(y + yd) * width + (x + xd)].isMine) then
												n := n + 1;
									end;
							gBoard^.tiles[y * width + x].neighbors := n;
						end;
				end;

		SetPt(newWindowSize, width * 16 + 20, height * 16 + 32 + 10);

		if ((newWindowSize.h <> gWindowSize.h) or (newWindowSize.v <> gWindowSize.v)) then
			begin
				SizeWindow(gWindow, newWindowSize.h, newWindowSize.v, true);

				y := 19 + (gScreenBounds.bottom - gScreenBounds.top) div 2 - (height * 16 + 32 + 10) div 2;
				x := (gScreenBounds.right - gScreenBounds.left) div 2 - (width * 16 + 20) div 2;

				MoveWindow(gWindow, x, y, true);

				gWindowSize := newWindowSize;

				SetEmptyRgn(gTmpRgn);
				RectRgn(gBackgroundRgn, gWindow^.portRect);

				SetRect(r, 10, 32, 11 + width * 16, 33 + height * 16);
				RectRgn(gTmpRgn, r);
				XorRgn(gTmpRgn, gBackgroundRgn, gBackgroundRgn);

				invalidate := true;
			end;

		if invalidate then
			begin
				GetPort(savePort);
				SetPort(gWindow);
				InvalRect(gWindow^.portRect);
				SetPort(savePort);
			end;
	end;

	procedure ResetScores;
		var
			i: Integer;
	begin
		for i := 0 to 2 do
			begin
				gHighScores[i].time := 99999;
				gHighScores[i].name := gDefaultName;
			end;
	end;

	procedure LoadScores;
		var
			pb: ParamBlockRec;
			err: OSErr;
	begin
		pb.ioNamePtr := @gScoresName;
		pb.ioVRefNum := gScoresVRefNum;
		pb.ioPermssn := fsRdPerm;
		err := PBOpenSync(@pb);
		if err = noErr then
			begin
				pb.ioBuffer := @gHighscores;
				pb.ioReqCount := SizeOf(Scores);
				pb.ioPosMode := fsFromStart;
				pb.ioPosOffset := 0;
				err := PBReadSync(@pb);
				err := PBCloseSync(@pb);
			end
		else
			ResetScores;
	end;

	procedure SaveScores;
		var
			pb: ParamBlockRec;
			err: OSErr;
	begin
		pb.ioNamePtr := @gScoresName;
		pb.ioVRefNum := gScoresVRefNum;
		err := PBCreateSync(@pb);
		pb.ioPermssn := fsWrPerm;
		err := PBOpenSync(@pb);
		if err = noErr then
			begin
				pb.ioBuffer := @gHighscores;
				pb.ioReqCount := SizeOf(Scores);
				pb.ioPosMode := fsFromStart;
				pb.ioPosOffset := 0;
				err := PBWriteSync(@pb);
				err := PBCloseSync(@pb);
			end
	end;

	procedure InitGlobals;
		var
			i: integer;
			p: Ptr;
			r: Rect;
			wmgrPort: GrafPtr;
			iconH: Handle;
			err: OSErr;
			sysEnv: SysEnvRec;
			s: Str255;
	begin
		err := SysEnvirons(1, sysEnv);
		gScoresVRefNum := sysEnv.sysVRefNum;

		SetPt(gWindowSize, 0, 0);

		if gColorQD then
			begin
				StuffHex(@gNeighborColors[1], '00000000FFFF');
				StuffHex(@gNeighborColors[2], '0000AAAA0000');
				StuffHex(@gNeighborColors[3], 'AAAAAAAA0000');
				StuffHex(@gNeighborColors[4], 'CCCC66660000');
				StuffHex(@gNeighborColors[5], 'FFFF00000000');
				StuffHex(@gNeighborColors[6], '888800004444');
				StuffHex(@gNeighborColors[7], '444400008888');
				StuffHex(@gNeighborColors[8], '000000000000');
			end;

		for i := 0 to 2 do
			begin
				GetIndString(s, OwnedResourceID(0), i + 1);
				gModeNames[i] := s;
			end;
		GetIndString(gDefaultName, OwnedResourceID(0), 4);
		GetIndString(gScoresName, OwnedResourceID(0), 5);

		gAlreadyOpen := true;
		gDone := false;
		gMenu := nil;
		gWindow := nil;
		gBoardType := kKindBeginner;
		gCustomWidth := 16;
		gCustomHeight := 16;
		gCustomMines := 40;

		LoadScores;

		StuffHex(Ptr(@gBackPat), '1100440011004400');
		StuffHex(Ptr(@gWhitePat), '0000000000000000');

		GetWMgrPort(wmgrPort);
		gScreenBounds := wmgrPort^.portRect;

		gTiles := GetResource('SICN', OwnedResourceID(0));
		SetRect(r, 0, 0, 16, 16);
		for i := 1 to 7 do
			begin
				p := NewBitMap(gTileBitmaps[i], r);
				BlockMove(Ptr(Ord(gTiles^) + ((i - 1) * 16 * 16 div 8)), p, 16 * 16 div 8);
				if gColorQD then
					gTileIcons[i] := GetCIcon(OwnedResourceID(i - 1));
			end;

		SetRect(r, 0, 0, 32, 32);
		for i := 1 to 8 do
			begin
				iconH := GetResource('ICON', OwnedResourceID(i - 1));
				p := NewBitMap(gButtonBitmaps[i], r);
				BlockMove(iconH^, p, 32 * 32 div 8);
				ReleaseResource(iconH);
				if gColorQD then
					gButtonIcons[i] := GetCIcon(OwnedResourceID(i + 6));
			end;

		iconH := GetResource('ICON', OwnedResourceID(8));
		p := NewBitMap(gButtonMask, r);
		BlockMove(iconH^, p, 32 * 32 div 8);
		ReleaseResource(iconH);

		ReleaseResource(gTiles);

		gBackgroundRgn := NewRgn;
	end;

	procedure CreateMenu;
	begin
		gMenuID := OwnedResourceID(kMenuID);
		gMenu := GetMenu(gMenuID);
		gMenu^^.menuID := gMenuID;
		InsertMenu(gMenu, 0);
		gDCE^.dCtlMenu := gMenuID;
		DrawMenuBar;
	end;

	procedure CreateWindow;
		const
			windowWidth = 9 * 16 + 20;
			windowHeight = 9 * 16 + 32 + 10;
		var
			windowTop, windowLeft: Integer;
			windowBottom, windowRight: Integer;
			savePort: GrafPtr;
			bounds: Rect;
	begin
		GetPort(savePort);
		windowTop := 19 + (gScreenBounds.bottom - gScreenBounds.top) div 2 - windowHeight div 2;
		windowLeft := (gScreenBounds.right - gScreenBounds.left) div 2 - windowWidth div 2;
		windowBottom := windowTop + windowHeight;
		windowRight := windowLeft + windowWidth;
		SetRect(bounds, windowLeft, windowTop, windowRight, windowBottom);

		if gColorQD then
			gWindow := NewCWindow(nil, bounds, 'Minesweeper', false, 4, nil, true, 0)
		else
			gWindow := NewWindow(nil, bounds, 'Minesweeper', false, 4, nil, true, 0);

		WindowPeek(gWindow)^.windowKind := gDCE^.dCtlRefNum;
		gDCE^.dCtlWindow := Pointer(gWindow);

		SetPort(gWindow);
		TextFont(geneva);
		TextSize(9);

		SetPort(savePort);
	end;

	procedure DoOpen;
	begin
		if gWindow <> nil then
			SelectWindow(gWindow);

		if gAlreadyOpen then
			gDCE^.dCtlMenu := gMenuID
		else
			begin
				GetWaitCursor;
				InitColor;
				InitGlobals;
				CreateMenu;
				CreateWindow;
				NewBoard(true);
				HiliteMenu(0);
			end
	end;

	procedure DestroyWindow;
	begin
		if gWindow <> nil then
			begin
				DisposeWindow(gWindow);
				gWindow := nil;
			end;
		gDCE^.dCtlWindow := nil;
	end;

	procedure DoClose;
		var
			i: Integer;
	begin
		DeleteMenu(gDCE^.dCtlMenu);
		DrawMenuBar;
		gDCE^.dCtlMenu := 0;
		DisposeMenu(gMenu);
		gMenu := nil;

		DestroyWindow;
		gAlreadyOpen := false;

		if (gBoard <> nil) then
			DisposePtr(Ptr(gBoard));

		for i := 1 to 8 do
			begin
				if gColorQD and (gButtonIcons[i] <> nil) then
					DisposeCIcon(gButtonIcons[i]);
				DisposePtr(gButtonBitmaps[i].baseAddr);
			end;

		for i := 1 to 7 do
			begin
				if gColorQD and (gTileIcons[i] <> nil) then
					DisposeCIcon(gTileIcons[i]);
				DisposePtr(gTileBitmaps[i].baseAddr);
			end;

		DisposePtr(gButtonMask.baseAddr);

		DisposeRgn(gBackgroundRgn);
		DisposeRgn(gTmpRgn);
	end;

	procedure DoDrawTile (x, y: Integer);
		var
			s: Str255;
			tileType: integer;
			r: Rect;
	begin
		SetRect(r, 11 + 16 * x, 33 + 16 * y, 27 + 16 * x, 49 + 16 * y);
		if (gBoard^.tiles[y * gBoard^.width + x].isRevealed) then
			begin
				if (gBoard^.tiles[y * gBoard^.width + x].isMine) then
					begin
						if (gBoard^.tiles[y * gBoard^.width + x].isTriggered) then
							tileType := kTileMineTriggered
						else if (gBoard^.tiles[y * gBoard^.width + x].flagType = 1) then
							tileType := kTileFlagged
						else
							tileType := kTileMine
					end
				else if (gBoard^.tiles[y * gBoard^.width + x].flagType = 1) then
					tileType := kTileNoMine
				else
					tileType := kTileRevealed
			end
		else if (gBoard^.tiles[y * gBoard^.width + x].flagType = 1) then
			tileType := kTileFlagged
		else if (gBoard^.tiles[y * gBoard^.width + x].flagType = 2) then
			tileType := kTilePossibly
		else
			begin
				if (gTrackingTile) then
					tileType := kTileRevealed
				else
					tileType := kTileNormal;
			end;
		if gColorQD then
			PlotCIcon(r, gTileIcons[tileType])
		else
			CopyBits(gTileBitMaps[tileType], gWindow^.portBits, gTileBitMaps[tileType].bounds, r, srcCopy, nil);
		if (gBoard^.tiles[y * gBoard^.width + x].isRevealed) then
			if (not gBoard^.tiles[y * gBoard^.width + x].isMine) then
				if (gBoard^.tiles[y * gBoard^.width + x].flagType <> 1) then
					if (gBoard^.tiles[y * gBoard^.width + x].neighbors > 0) then
						begin
							MoveTo(r.left + 4, r.top + 11);
							NumToString(gBoard^.tiles[y * gBoard^.width + x].neighbors, s);
							if gColorQD then
								RGBForeColor(gNeighborColors[gBoard^.tiles[y * gBoard^.width + x].neighbors]);
							TextFont(geneva);
							TextSize(9);
							TextFace([bold]);
							TextMode(srcOr);
							DrawString(s);
							ForeColor(blackColor);
						end;
	end;

	procedure DrawTile (x, y: Integer);
		var
			savePort: GrafPtr;
	begin
		GetPort(savePort);
		SetPort(gWindow);
		DoDrawTile(x, y);
		SetPort(savePort);
	end;

	procedure DrawMineCount;
		var
			r: Rect;
			savePort: GrafPtr;
			i: Integer;
			s: Str255;
			rgb: RGBColor;
	begin
		GetPort(savePort);
		SetPort(gWindow);

		SetRect(r, 10, 3, 58, 29);
		OffsetRect(r, 1, 1);
		if gColorQD then
			begin
				rgb.red := $6666;
				rgb.green := $6666;
				rgb.blue := $6666;
				RGBForeColor(rgb);
			end;
		PenSize(1, 1);
{FrameRect(r);}
		OffsetRect(r, -1, -1);
		ForeColor(blackColor);
		FrameRect(r);
		PenNormal;
		InsetRect(r, 1, 1);
		EraseRect(r);
		TextFont(0);
		TextSize(12);
		TextFace([]);
		TextMode(srcOr);
		NumToString(gFlagCounter, s);
		MoveTo((r.left + r.right) div 2 - StringWidth(s) div 2, r.bottom - 7);
		DrawString(s);

		SetPort(savePort);
	end;

	procedure DrawTimer;
		var
			r: Rect;
			savePort: GrafPtr;
			i: Integer;
			s: Str255;
			rgb: RGBColor;
	begin
		GetPort(savePort);
		SetPort(gWindow);

		SetRect(r, gWindowSize.h - 58, 3, gWindowSize.h - 9, 29);
		OffsetRect(r, 1, 1);
		if gColorQD then
			begin
				rgb.red := $6666;
				rgb.green := $6666;
				rgb.blue := $6666;
				RGBForeColor(rgb);
			end;
		PenSize(1, 1);
{FrameRect(r);}
		OffsetRect(r, -1, -1);
		ForeColor(blackColor);
		FrameRect(r);
		PenNormal;
		InsetRect(r, 1, 1);
		EraseRect(r);
		TextFont(0);
		TextSize(12);
		TextFace([]);
		TextMode(srcOr);
		NumToString(gTimerTicks div 60, s);
		MoveTo((r.left + r.right) div 2 - StringWidth(s) div 2, r.bottom - 7);
		DrawString(s);

		SetPort(savePort);
	end;

	procedure DrawFace (hilite: Boolean);
		var
			r: Rect;
			savePort: GrafPtr;
			i: Integer;
	begin
		GetPort(savePort);
		SetPort(gWindow);

		SetRect(r, 0, 0, 32, 32);
		OffsetRect(r, 10 + (gBoard^.width * 16 div 2) - 16, 0);
		if (gGameOver) then
			i := 3
		else if (gWin) then
			i := 7
		else if (gTrackingTile) then
			i := 5
		else
			i := 1;

		if (hilite) then
			i := i + 1;

		if gColorQD then
			PlotCIcon(r, gButtonIcons[i])
		else
			CopyMask(gButtonBitmaps[i], gButtonMask, gWindow^.portBits, gButtonBitmaps[i].bounds, gButtonMask.bounds, r);

		SetPort(savePort);
	end;

	procedure DrawBoard;
		var
			savePort: GrafPtr;
			x, y: integer;
	begin
		GetPort(savePort);
		SetPort(gWindow);
		for y := 0 to gBoard^.height - 1 do
			for x := 0 to gBoard^.width - 1 do
				DoDrawTile(x, y);
		SetPort(savePort);
	end;

	procedure ReDrawBoard (bw: Boolean);
		var
			savePort: GrafPtr;
			r: Rect;
			fg, bk: RGBColor;
			i: Integer;
	begin
		GetPort(savePort);
		SetPort(gWindow);

		if not bw then
			begin
				bk.red := $DADA;
				bk.green := $DADA;
				bk.blue := $DADA;
				fg.red := $FFFF;
				fg.green := $FFFF;
				fg.blue := $FFFF;
				RGBForeColor(fg);
				RGBBackColor(bk);
			end;
		BackPat(gBackPat);
		EraseRgn(gBackgroundRgn);
		BackPat(gWhitePat);
		ForeColor(blackColor);
		BackColor(whiteColor);

		DrawFace(false);
		DrawMineCount;
		DrawTimer;

		SetRect(r, 10, 32, 11 + 16 * gBoard^.width, 33 + 16 * gBoard^.height);
		FrameRect(r);

		DrawBoard;

		SetPort(savePort);
	end;

	procedure DrawBoardProc (depth: Integer; deviceFlags: Integer; targetDevice: GDHandle; userData: LongInt);
		var
			bw: Boolean;
	begin
		if depth < 4 then
			bw := true
		else
			bw := false;
		ReDrawBoard(bw);
	end;

	procedure UpdateMenuChecks;
	begin
		SetItemMark(gMenu, 3, ' ');
		SetItemMark(gMenu, 4, ' ');
		SetItemMark(gMenu, 5, ' ');
		SetItemMark(gMenu, 6, ' ');
		SetItemMark(gMenu, gBoardType + 3, Char(18));
	end;

	procedure SetDialogScores (d: DialogPtr);
		var
			s: Str255;
			i: Integer;
	begin
		for i := 0 to 2 do
			begin
				NumToString(gHighScores[i].time, s);
				s := concat(s, ' seconds');
				SetStrItem(d, 2 + i * 2, s);
				SetStrItem(d, 2 + i * 2 + 1, gHighScores[i].name);
			end;
	end;

	procedure Highscores;
		var
			savePort: GrafPtr;
			d: DialogPtr;
			item: Integer;
	begin
		d := CenterNewDialog(OwnedResourceID(3), nil, Pointer(-1));
		GetPort(savePort);
		SetPort(d);
		SetDialogScores(d);
		item := 0;
		while (item <> 1) do
			begin
				ModalDialog(@SFFilter, item);
				if item = 8 then
					begin
						ResetScores;
						SetDialogScores(d);
						SaveScores;
					end;
			end;
		DisposDialog(d);
		SetPort(savePort);
	end;

	procedure CheckHighscore;
		var
			score: Integer;
			savePort: GrafPtr;
			d: DialogPtr;
			item: Integer;
			s: Str255;
	begin
		if gBoardType = kKindCustom then
			Exit(CheckHighscore);
		score := gTimerticks div 60;
		if score < gHighScores[gBoardType].time then
			begin
				ParamText(gModeNames[gBoardType], gHighScores[gBoardType].name, '', '');
				d := CenterNewDialog(OwnedResourceID(2), nil, Pointer(-1));
				SetStrItem(d, 2, '');
				GetPort(savePort);
				SetPort(d);
				item := 0;
				while item <> 1 do
					begin
						ModalDialog(@SFFilter, item);
					end;
				ReadStrItem(d, 2, s);
				gHighScores[gBoardType].name := s;
				gHighScores[gBoardType].time := score;
				SaveScores;
				DisposDialog(d);
				SetPort(savePort);
				Highscores;
			end;
	end;

	procedure CustomizeGame;
		const
			kItemOK = 1;
			kItemCancel = 2;
			kItemWidth = 3;
			kItemHeight = 4;
			kItemMines = 5;
		var
			savePort: GrafPtr;
			d: DialogPtr;
			item: Integer;
			l: LongInt;
			s: Str255;
	begin
		d := CenterNewDialog(OwnedResourceID(1), nil, Pointer(-1));
		GetPort(savePort);
		SetPort(d);

		NumToString(gCustomWidth, s);
		SetStrItem(d, kItemWidth, s);
		NumToString(gCustomHeight, s);
		SetStrItem(d, kItemHeight, s);
		NumToString(gCustomMines, s);
		SetStrItem(d, kItemMines, s);

		item := 0;
		while ((item <> 1) and (item <> 2)) do
			begin
				ModalDialog(@SFFilter, item);
			end;

		if (item = 1) then
			begin
				ReadStrItem(d, kItemWidth, s);
				StringToNum(s, l);
				if (l < 8) then
					l := 8
				else if (l > 30) then
					l := 30;
				gCustomWidth := Integer(l);

				ReadStrItem(d, kItemHeight, s);
				StringToNum(s, l);
				if (l < 8) then
					l := 8
				else if (l > 16) then
					l := 16;
				gCustomHeight := Integer(l);

				ReadStrItem(d, kItemMines, s);
				StringToNum(s, l);
				if (l < 10) then
					l := 10
				else if (l > 3 * gCustomWidth * gCustomHeight div 4) then
					l := 3 * gCustomWidth * gCustomHeight div 4;
				gCustomMines := Integer(l);

				gBoardType := kKindCustom;
				UpdateMenuChecks;
			end;

		DisposDialog(d);
		SetPort(savePort);

		InitCursor;

		if (item = 1) then
			begin
				NewBoard(true);
			end;
	end;

	procedure DoMenu (i: integer);
	begin
		HiliteMenu(gDCE^.dCtlMenu);
		case i of
			1: 
				begin
					NewBoard(true);
				end;
			3..5: 
				begin
					gBoardType := i - 3;
					UpdateMenuChecks;
					NewBoard(true);
				end;
			6: 
				CustomizeGame;
			8: 
				Highscores;
		end;
		HiliteMenu(0);
	end;

	procedure DoDeviceLoop (drawingRgn: RgnHandle; drawingProc: DeviceLoopDrawingProcPtr; userData, flags: LongInt);
	inline
		$ABCA;

	procedure HandleUpdate (event: EventRecord);
		var
			savePort: GrafPtr;
			w: WindowPeek;
			proc: DeviceLoopDrawingProcPtr;
	begin
		w := WindowPeek(event.message);

		if (w = WindowPeek(gWindow)) then
			begin
				GetPort(savePort);
				SetPort(gWindow);

				BeginUpdate(gWindow);
				if gColorQD then
					begin
						proc := @DrawBoardProc;
						DoDeviceLoop(gWindow^.visRgn, proc, 0, 0);
					end
				else
					ReDrawBoard(true);
				DrawControls(gWindow);
				EndUpdate(gWindow);

				SetPort(savePort);
			end;
	end;

	procedure HandleActivate (event: EventRecord);
		var
			savePort: GrafPtr;
			activate: Boolean;
	begin
		activate := Boolean(BitAnd(event.modifiers, activeFlag));
		GetPort(savePort);
		SetPort(gWindow);
		SetPort(savePort);
	end;

	procedure RevealMines;
		var
			x, y: Integer;
	begin
		for y := 0 to gBoard^.height - 1 do
			for x := 0 to gBoard^.width - 1 do
				begin
					if (gBoard^.tiles[y * gBoard^.width + x].isMine) then
						gBoard^.tiles[y * gBoard^.width + x].isRevealed := true
					else if (gBoard^.tiles[y * gBoard^.width + x].flagType = 1) then
						gBoard^.tiles[y * gBoard^.width + x].isRevealed := true;
				end;
		DrawBoard;
	end;

	procedure CheckWin;
		var
			x, y: Integer;
			anyUnrevealedNonMines: Boolean;
	begin
		if (gGameOver or gWin) then
			Exit(CheckWin);

		for y := 0 to gBoard^.height - 1 do
			for x := 0 to gBoard^.width - 1 do
				begin
					if (not gBoard^.tiles[y * gBoard^.width + x].isRevealed) then
						if (not gBoard^.tiles[y * gBoard^.width + x].isMine) then
							anyUnrevealedNonMines := true;
				end;

		if (not anyUnrevealedNonMines) then
			begin
				gWin := true;
				DrawFace(false);
				for y := 0 to gBoard^.height - 1 do
					for x := 0 to gBoard^.width - 1 do
						begin
							if (not gBoard^.tiles[y * gBoard^.width + x].isRevealed) then
								if (gBoard^.tiles[y * gBoard^.width + x].isMine) and (gBoard^.tiles[y * gBoard^.width + x].flagType <> 1) then
									begin
										gBoard^.tiles[y * gBoard^.width + x].flagType := 1;
										gFlagCounter := gFlagCounter - 1;
										DrawTile(x, y);
									end;
						end;
				DrawMineCount;
				CheckHighscore;
			end;
	end;

	procedure RevealAdjancedTiles;
		var
			x, y: Integer;
			xd, yd: Integer;
			reveal: Boolean;
			continue: Boolean;
	begin
		continue := true;
		while (continue) do
			begin
				continue := false;
				for y := 0 to gBoard^.height - 1 do
					for x := 0 to gBoard^.width - 1 do
						if (not gBoard^.tiles[y * gBoard^.width + x].isRevealed) then
							if (not gBoard^.tiles[y * gBoard^.width + x].isMine) then
								begin
									reveal := false;
									for yd := -1 to 1 do
										for xd := -1 to 1 do
											begin
												if (InRange(x + xd, y + yd)) then
													if (gBoard^.tiles[(y + yd) * gBoard^.width + (x + xd)].isRevealed) then
														if (not gBoard^.tiles[(y + yd) * gBoard^.width + (x + xd)].isMine) then
															if (gBoard^.tiles[(y + yd) * gBoard^.width + (x + xd)].neighbors = 0) then
																reveal := true;
											end;
									if (reveal) then
										begin
											gBoard^.tiles[y * gBoard^.width + x].floodfill := true;
											continue := true;
										end;
								end;
				for y := 0 to gBoard^.height - 1 do
					for x := 0 to gBoard^.width - 1 do
						if gBoard^.tiles[y * gBoard^.width + x].floodfill then
							begin
								gBoard^.tiles[y * gBoard^.width + x].isRevealed := true;
								gBoard^.tiles[y * gBoard^.width + x].floodfill := false;
								DrawTile(x, y);
							end;
			end;
		CheckWin;
	end;

	procedure TrackFace;
		var
			p: Point;
			state, inside: Boolean;
			r: Rect;
			savePort: GrafPtr;
	begin
		GetPort(savePort);
		SetPort(gWindow);

		SetRect(r, 0, 0, 32, 32);
		OffsetRect(r, 10 + (gBoard^.width * 16 div 2) - 16, 0);

		state := true;
		DrawFace(true);

		while Button do
			begin
				GetMouse(p);
				inside := PtInRect(p, r);
				if (inside and not state) then
					DrawFace(true)
				else if (not inside and state) then
					DrawFace(false);
				state := inside;
			end;

		DrawFace(false);
		SetPort(savePort);

		if (state) then
			begin
				NewBoard(true);
			end;
	end;

	function TrackTile (var x, y: Integer): Boolean;
		var
			savePort: GrafPtr;
			r: Rect;
			p: Point;
			mx, my: Integer;
			state, inside: Boolean;
	begin
		GetPort(savePort);
		SetPort(gWindow);

		gTrackingTile := true;
		state := true;
		DrawTile(x, y);

		DrawFace(false);

		while (Button) do
			begin
				GetMouse(p);
				mx := (p.h - 10) div 16;
				my := (p.v - 32) div 16;
				inside := ((mx = x) and (my = y) and (p.h >= 10) and (p.v >= 32));
				if (inside <> state) then
					begin
						gTrackingTile := inside;
						DrawTile(x, y);
						state := inside;
						gTrackingTile := true;
						if (InRange(mx, my)) then
							begin
								x := mx;
								y := my;
								DrawTile(x, y);
								state := true;
							end;
					end;
			end;

		gTrackingTile := false;
		SetPort(savePort);

		DrawFace(false);
		DrawTile(x, y);

		TrackTile := state;
	end;

	procedure DoMouse (p: Point; cmd: Boolean);
		var
			x, y, f: Integer;
			savePort: GrafPtr;
			r: Rect;
	begin
		GetPort(savePort);
		SetPort(gWindow);
		GlobalToLocal(p);
		SetPort(savePort);

		SetRect(r, 0, 0, 32, 32);
		OffsetRect(r, 10 + (gBoard^.width * 16 div 2) - 16, 0);
		if (PtInRect(p, r)) then
			begin
				TrackFace;
			end;

		x := (p.h - 10) div 16;
		y := (p.v - 32) div 16;
		if ((not InRange(x, y)) or (p.h < 10) or (p.v < 32)) then
			Exit(DoMouse);

		if (gGameOver) then
			Exit(DoMouse);

		if not gGameStarted then
			begin
				gGameStarted := true;
				gTimerTicks := 60;
				DrawTimer;
			end;

		if (cmd) then
			begin
				if (not gBoard^.tiles[y * gBoard^.width + x].isRevealed) then
					begin
						f := gBoard^.tiles[y * gBoard^.width + x].flagType;
						f := f + 1;
						case f of
							1: 
								gFlagCounter := gFlagCounter - 1;
							2: 
								gFlagCounter := gFlagCounter + 1;
							otherwise
								f := 0;
						end;
						gBoard^.tiles[y * gBoard^.width + x].flagType := f;
						DrawTile(x, y);
						DrawMineCount;
					end;
			end
		else if (not gBoard^.tiles[y * gBoard^.width + x].isRevealed) then
			if (gBoard^.tiles[y * gBoard^.width + x].flagType = 0) then
				if (TrackTile(x, y)) then
					begin
						gBoard^.tiles[y * gBoard^.width + x].isRevealed := true;
						if (gBoard^.tiles[y * gBoard^.width + x].isMine) then
							begin
								gBoard^.tiles[y * gBoard^.width + x].isTriggered := true;
								gGameOver := true;
								RevealMines;
								DrawFace(false);
							end;
						DrawTile(x, y);
						RevealAdjancedTiles;
					end;
	end;

	procedure DoIdle;
		var
			lastSec: Integer;
	begin
		if (not gGameOver and not gWin and gGameStarted) then
			begin
				lastSec := gTimerTicks div 60;
				gTimerTicks := gTimerTicks + 1;
				if ((gTimerTicks div 60) <> lastSec) then
					DrawTimer;
			end;
	end;

	procedure DoEvent (e: EventRecord);
	begin
		case e.what of
			updateEvt: 
				HandleUpdate(e);
			mouseDown: 
				DoMouse(e.where, Boolean(BitAnd(e.modifiers, cmdKey) <> 0));
			activateEvt: 
				HandleActivate(e);
			keyDown, autoKey: 
				;
			otherwise
				;
		end;
	end;

	procedure DoGoodBye;
	begin
	end;

	procedure UpdateMenuBar;
	begin
	end;

	procedure DoControl (p: ParmBlkPtr);
		type
			EventPtr = ^EventRecord;
	begin
		case p^.csCode of
			accMenu: 
				DoMenu(p^.csParam[1]);
			accEvent: 
				DoEvent(EventPtr(p^.ioMisc)^);
			accRun: 
				DoIdle;
			goodBye: 
				DoGoodBye;
			otherwise
				;
		end;

		if p^.csCode <> accCursor then
			UpdateMenuBar;
	end;

	function Main (d: DCtlPtr; p: ParmBlkPtr; n: integer): OSErr;
		const
			TheOpen = 0;
			ThePrime = 1;
			TheControl = 2;
			TheStatus = 3;
			TheClose = 4;
			dCtlEnable = $0400;

		var
			oldFlags: integer;

	begin
{$IFC NOT DAShell}
		RememberA4;
{$ENDC}
		gDCE := d;

{$IFC DAShell}
		if n = TheOpen then
			begin
				gDCE^.dCtlFlags := BitOr(gDCE^.dCtlFlags, dNeedTime);
				gDCE^.dCtlDelay := 0;
				gDCE^.dCtlCurTicks := 0;
			end;
{$ENDC}

		oldFlags := gDCE^.dCtlFlags;
		gDCE^.dCtlFlags := BitAnd(gDCE^.dCtlFlags, BitNot(dCtlEnable));

		case n of
			TheOpen: 
				begin
					if d^.dCtlStorage = nil then
						begin
							SysBeep(2);
							Main := -108;
						end;
					DoOpen;
				end;

			TheControl: 
				DoControl(p);

			TheClose: 
				gDone := true;

			otherwise
				;
		end;

		if gAlreadyOpen and gDone then
			DoClose;

		gDCE^.dCtlFlags := oldFlags;
		Main := noErr;
	end;

end.